///////////////////////////////////////////////////////////////////////////////////////
//
// MonoScan
//
// 27/04/07		Hounddog	Initial implementation
//

namespace MonoScan
{
	using System;
	using System.Runtime.InteropServices;
	using System.Collections;

	/// <summary>
	/// This class is a .NET proxy for MonoComm.dll. Deals with data marshalling and memory management.
	/// See MonoComm.h for the detailed description of each function.
	/// </summary>
	[StructLayout(LayoutKind.Auto, CharSet = CharSet.Ansi)]
	public class MonoComm
	{
		private MonoComm() {}

		public delegate void MessageHandlerDelegate(IntPtr contextPtr, string message);	// STDCALL is default

		public enum Status : uint
		{
			Ok							= 0,
			Error						= 0xFFFFFFFF,
			ErrorImpl					= 0xFFFFFFFE,
			ErrorEcu					= 0xFFFFFFFD,
			ErrorConnect				= 0xFFFFFFFC,
			ErrorConnectPort			= 0xFFFFFFFB,
			ErrorConnectEcu				= 0xFFFFFFFA,
			ErrorConnectEcuEcho			= 0xFFFFFFF9,
			ErrorConnectAdapter			= 0xFFFFFFF8,
			ErrorConnectBlockIndex		= 0xFFFFFFF7
		}

		public enum MessageLevel : uint
		{
			None,
			Error,
			Log0,
			Log1,
			Log2
		}

		[DllImport("MonoComm.dll", EntryPoint = "McConstruct", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern Status Construct(IntPtr contextPtr, MessageHandlerDelegate messageHandlerDelegate);

		[DllImport("MonoComm.dll", EntryPoint = "McDestruct", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern void Destruct();

		[DllImport("MonoComm.dll", EntryPoint = "McSetMessagesLevel", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern void SetMessageLevel(MessageLevel messageLevel);

		[DllImport("MonoComm.dll", EntryPoint = "McGetMessagesLevel", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern MessageLevel GetMessageLevel();

		public static Status Connect(string portName, uint baudRate, byte address, out string[] descriptions)
		{
			IntPtr descriptionsPtr;
			Status status = Connect(portName, baudRate, address, out descriptionsPtr);

			if (status == Status.Ok)
			{
				ArrayList arrayList = new ArrayList();

				IntPtr descriptionPtr;
				while ((descriptionPtr = Marshal.ReadIntPtr(descriptionsPtr, arrayList.Count * IntPtr.Size)) != IntPtr.Zero)
				{
					arrayList.Add(Marshal.PtrToStringAnsi(descriptionPtr));
					Free(descriptionPtr);
				}

				Free(descriptionsPtr);

				descriptions = (string[]) arrayList.ToArray(typeof(string));
			}
			else
				descriptions = null;

			return status;
		}

		[DllImport("MonoComm.dll", EntryPoint = "McConnect", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern Status Connect(string portName, uint baudRate, byte address, out IntPtr descriptionsPtr);

		[DllImport("MonoComm.dll", EntryPoint = "McDisconnect", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern void Disconnect();

		[DllImport("MonoComm.dll", EntryPoint = "McGetMaxIdleInterval", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern uint GetMaxIdleInterval();

		[DllImport("MonoComm.dll", EntryPoint = "McIdle", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern Status Idle();

		public static Status GetDescriptions(out string[] descriptions)
		{
			IntPtr descriptionsPtr;
			Status status = GetDescriptions(out descriptionsPtr);

			if (status == Status.Ok)
			{
				ArrayList arrayList = new ArrayList();

				IntPtr descriptionPtr;
				while ((descriptionPtr = Marshal.ReadIntPtr(descriptionsPtr, arrayList.Count * IntPtr.Size)) != IntPtr.Zero)
				{
					arrayList.Add(Marshal.PtrToStringAnsi(descriptionPtr));
					Free(descriptionPtr);
				}

				Free(descriptionsPtr);

				descriptions = (string[]) arrayList.ToArray(typeof(string));
			}
			else
				descriptions = null;

			return status;
		}

		[DllImport("MonoComm.dll", EntryPoint = "McGetDescriptions", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern Status GetDescriptions(out IntPtr descriptionsPtr);

		[StructLayoutAttribute(LayoutKind.Sequential, Pack = 1, Size = 8)]
		public struct Fault
		{
			public bool Sporadic;
			public ushort Code;
			public byte SubCode;			
		}

		public static Status GetFaults(out Fault[] faults)
		{
			uint count;
			IntPtr faultsPtr;
			Status status = GetFaults(out count, out faultsPtr);

			if (status == Status.Ok)
			{
				faults = new Fault[count];

				Type faultType = typeof(Fault);

				for (int index = 0; index < count; ++ index)
				{
					IntPtr faultPtr = (IntPtr) ((int) faultsPtr + index * Marshal.SizeOf(faultType));
					faults[index] = (Fault) Marshal.PtrToStructure(faultPtr, faultType);
				}

				Free(faultsPtr);
			}
			else
				faults = null;

			return status;
		}

		[DllImport("MonoComm.dll", EntryPoint = "McGetFaults", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern Status GetFaults(out uint count, out IntPtr faultsPtr);

		[DllImport("MonoComm.dll", EntryPoint = "McClearFaults", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern Status ClearFaults();

		public static Status GetBasicSettingsVar(out byte[] zones)
		{
			uint count;
			IntPtr zonesPtr;
			Status status = GetBasicSettingsVar(out count, out zonesPtr);

			if (status == Status.Ok)
			{
				zones = new byte[count];
				Marshal.Copy(zonesPtr, zones, 0, zones.Length);

				Free(zonesPtr);
			}
			else
				zones = null;

			return status;
		}

		[DllImport("MonoComm.dll", EntryPoint = "McGetBasicSettingsVar", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern Status GetBasicSettingsVar(out uint count, out IntPtr zonesPtr);

		public static Status GetBasicSettings(byte[] zones)
		{
			return GetBasicSettings((uint) zones.Length, zones);
		}

		// Do you feel the difference between out and [Out] ? :)
		[DllImport("MonoComm.dll", EntryPoint = "McGetBasicSettings", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern Status GetBasicSettings(uint count, [Out] byte[] zones);

		[DllImport("MonoComm.dll", EntryPoint = "McTestActuator", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		public static extern Status TestActuator(out ushort actuatorCode);

		[DllImport("MonoComm.dll", EntryPoint = "McAllocate", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern IntPtr Allocate(IntPtr size);

		[DllImport("MonoComm.dll", EntryPoint = "McFree", CallingConvention = CallingConvention.StdCall, ExactSpelling = true)]
		private static extern void Free(IntPtr buffer);

		public class Exception : System.Exception
		{
			public Exception(Status status) : base("MonoComm: " + StatusToString(status))
			{
				this.status = status;
			}

			public Status Status
			{
				get { return status; }
			}

			protected Status status;

			static string StatusToString(Status status)
			{
				switch (status)
				{
					case Status.Error:
						return "Error";

					case Status.ErrorImpl:
						return "Not implemented";

					case Status.ErrorEcu:
						return "ECU error";

					case Status.ErrorConnect:
						return "Communication error";

					case Status.ErrorConnectPort:
						return "Communication port error";

					case Status.ErrorConnectEcu:
						return "ECU timeout error";

					case Status.ErrorConnectEcuEcho:
						return "ECU echo error";

					case Status.ErrorConnectAdapter:
						return "Adapter timeout error";

					case Status.ErrorConnectBlockIndex:
						return "Block index error";

					default:
						return "0x" + status.ToString("X8");
				}
			}
		}
	}
}